/************************************************************/
/* SAS code: Example 9.10                                   */   
/* File: epls.sas                                           */  
/* This program (coded by Yingcun Xia) solves the model:    */
/*                                                          */
/* y = X^T beta + g( theta^T X) + epsilon                   */
/*                                                          */
/* IMPORTANT: DATA MUST BE STANDARDIZED BEFORE CALCULATION  */
/* Reference:                                               */
/* Xia, Y., Tong, H., and  Li, W.K. (1999).                 */
/*   On extended partially linear single-index models,      */
/*   Biometrika, 86(4), 831-842                             */
/*   DOI: 10.1093/biomet/86.4.831.                          */ 
/************************************************************/
PROC IML;

n  = 50;
x1 = normal(J(n, 1, -1));
x2 = normal(J(n, 1, -1));

y = 0.3*x1+0.4*x2+exp(-2*(0.8*x1-0.6*x2)##2)+0.1*normal(J(n,1,-1));
x = x1||x2;

/******************************************************/
/* Calculate parameters:			      */
/*      nitert: number of iterations for theta        */
/*               30 is used if not change             */ 
/*      niterb: number of iterations for beta         */
/*               10 is used if not change             */
/******************************************************/ 
options ps = 2000;
nitert     = 30;
niterb     = 10;
	
/*******************************************************/
START est(delet, h, p, n, xx, xxp, y, span, _ss0, _ye);
	n0 = nrow(xxp);
        _J1 = J(n, 1, 1);
	_ss0 = 0.0;
	FREE _ye0;
        DO _i = 1 TO n0;
        	_xy = xx - REPEAT(xxp[_i, ], n);
        	_x  = _xy#_xy*J(p,1,1);
        	_xx = _J1||_xy;
                _k  = EXP(- _x /(2*h*h) );
		IF delet = 1 THEN _k[_i] = 0.0; 
                _k   = _k/( SUM(_k) +1.0e-10 );
                _xy  = t(_k#_xx);
                _va  = _xy*_xx + 1.0E-5*I(p+1);
                _vb  = _xy*y;
                _a   = SOLVE(_va, _vb);
                _ss0 = _ss0 + (y[_i] - _a[1] )**2;
		_ye0 = _ye0//_a[1];
        END;
	_ss0 = _ss0/n;
	_ye  = _ye0;
FINISH est;

START cv(hL, hU, p, n, xx, y, span, _h0, _ss0);
	_J1 = J(n,1,1);
	_hL = hL;
	_hU = hU;
	
	_ss0 = 0.0;
	_hm  = _hL + (_hU - _hL)/2;
        DO _i = 1 TO n;
        	_xy = xx - REPEAT(xx[_i, ], n);
        	_x  = (_xy#_xy)*J(p,1,1);
		_x[_i] = 1.0E20;
        	_k   = EXP(- _x /(2*_hm*_hm) );
        	_k   = _k/( SUM(_k) +1.0e-10 );
                _xx  = _J1||_xy;
		_kxx = t(_k#_xx);
                _va  = _kxx*_xx + 1.0E-5*I(p+1);
                _vb  = _kxx*y;
                _a   = SOLVE(_va, _vb);
	      	_ss0 = _ss0 + (y[_i] - _a[1] )**2;
	END;

	DO _hi = 0 TO 10 WHILE ( _hU - _hL > 0.0001 );
		_h1  = _hL +  (_hU-_hL)/3;
		_h2  = _hL +2*(_hU-_hL)/3;
		_ss1 = 0.0;
        	_ss2 = 0.0;
        	DO _i = 1 TO n;
                _xy    = xx - REPEAT(xx[_i, ], n);
                _x     = (_xy#_xy)*J(p,1,1);
		_x[_i] = 1.0E20; 
                _k     = EXP(- _x /(2*_h1*_h1) );
        	_k     = _k/( SUM(_k) +1.0e-10 );
                _xx    = _J1||_xy;
		_kxx   = t(_k#_xx);
                _va    = _kxx*_xx + 1.0E-5*I(p+1);
                _vb    = _kxx*y;
                _a     = SOLVE(_va, _vb);
        	_ss1   = _ss1 + (y[_i] - _a[1] )**2;

                _k     = EXP(- _x /(2*_h2*_h2) );
        	_k     = _k/( SUM(_k) +1.0e-10 );
                _xx    = _J1||_xy;
		_kxx   = t(_k#_xx);
                _va    = _kxx*_xx + 1.0E-5*I(p+1);
                _vb    = _kxx*y;
                _a     = SOLVE(_va, _vb);
        	_ss2   = _ss2 + (y[_i] - _a[1] )**2;

        	END;
		
		_ss = MIN(_ss0, _ss1, _ss2);
		IF _ss1 = _ss THEN DO;
			_hU = _hm;
			_hm = _h1;
			END;
                IF _ss2 = _ss THEN DO;
                        _hL = _hm;
                        _hm = _h2;
                        END;
                IF _ss0 = _ss THEN DO;
                        _hU = _h2;
                        _hL = _h1;
                        END;
		_ss0 = _ss;
	END;
	_h0  = (_hU + _hL)/2;
	_ss0 = _ss0/n;
FINISH cv;

/**************************************/
xx = x;
n  = nrow(y);
p  = ncol(xx);

_beta0  = J(p, 1, 1);
_theta0 = J(p, 1, 1)/sqrt(p);
d1      = J(n, 1, 0);
d2      = J(n, p, 0);
a       = J(n, p, 0);
a1      = J(n, 1, 0);
a2      = J(n, p, 0);
c       = J(p, p, 0);
c1      = J(n, p, 0);
c2      = J(n, p*p, 0);
b2      = J(n, p*p, 0);

RUN cv(0.01, 5, p, n, xx, y, 1, h, _ss0);

DO _iterb = 1 TO niterb;
d = 0;
c = J(p, p, 0);

IF _iterb > 1 THEN RUN cv(0.01, 5, 1, n, xx*theta, y, 1, h, _ss0);
 
do j = 1 to n;
        xxj = xx - repeat(xx[j,], n);
        IF _iterb > 1 THEN;
        Kx = exp(-(xxj*theta)##2/(2*h*h));
        IF _iterb = 1 THEN
        Kx     = exp(-(xxj#xxj)*J(p,1,1)/(2*h*h));
        Kx     = Kx/sum(Kx);
        tKx    = t(Kx);
        d      = d + tKx * (xx # y);
        d1[j ] = tKx*y;
        d2[j,] = tKx*(xxj#y);
        a[j,] = tKx*xx;
        a1[j] = sum(tKx);
        a2[j, ] = tKx*xxj;
        c       = c + t(Kx#xx)*xx;
        c1[j,]  = tKx*xx;
        c2[j,]  = shape(t(Kx#xxj)*xx, 1, p*p);
        b2[j, ] = shape(t(Kx#xxj)*xxj, 1, p*p);
end;

_a = J(n, 1, 0);
_b = J(n, 1, 0);
do iter = 1 to nitert;
        _u = c;
        _v = t(d);
        do j = 1 to n;
                f = (a1[j]||(a2[j,]*_theta0))
                        //((a2[j,]*_theta0)
                        || t(_theta0)* shape(b2[j,], p, p)*_theta0);
                invf  = ginv(f);
                cc    = c1[j,]//t(_theta0)*shape(c2[j,], p, p);

                _tmp0 = d1[j]//(d2[j,]*_theta0);
                _tmp1 = (t(a[j,])*invf[1,] +
                               t(shape(c2[j,],p,p))*_theta0*invf[2,]);
                _u    = _u - _tmp1*cc;
                _v    = _v - _tmp1*_tmp0;
        end;
        _u = (_u||_theta0)//(t(_theta0)||0);
        _v = _v//0;
        _tmpb  = Ginv(_u)*_v;
        beta   = _tmpb[1:p];
        _beta0 = beta;

        do j = 1 to n;
                f = (a1[j]||(a2[j,]*_theta0))
                        //((a2[j,]*_theta0)
                          || t(_theta0)* shape(b2[j,], p, p)*_theta0);
                invf  = ginv(f);
                cc    = c1[j,]//t(_theta0)*shape(c2[j,], p, p);
                _tmp0 = d1[j]//(d2[j,]*_theta0);

                _ab   = invf*(_tmp0 - cc*_beta0);
                _a[j] = _ab[1];
                _b[j] = _ab[2];
        end;

        s2 = 0;
        s1 = 0;
        do j = 1 to n;
                s2 = s2 + shape(b2[j,], p, p)*_b[j]*_b[j];
                s1 = s1 + _b[j] *
                       (
                        t(d2[j,])
                        - shape(c2[j, ], p,p)*beta
                        - _a[j]*t(a2[j,])
                        );
        end;
        theta   = ginv(s2)*s1;
        theta   = theta/sqrt(t(theta)*theta);
        _theta0 = theta;
        _beta0  = beta;
end;

END;

xxp = xx*theta;
xx0 = xxp;
ye  = y - xx*beta;
RUN est(0, h, 1, n, xx0, xxp, ye, 1, _ss0, _ypred);
ypred = xx*beta + _ypred;
print beta;

beta   = (I(p) - theta*t(theta))*beta;
r      = rank(xxp);
txt    = J(n, 1, 0);
txt[r] = xxp;
xtheta = txt;
txt[r] = _ypred;
ytheta = txt;
txt[r] = ye;
yr     = txt;

/***************************************************/
/* PRINT OUTPUT:                                   */
/*          beta = (I-theta*theta^T)*beta          */
/*          theta = +/- theta                      */
/*  IMPORTANT: beta MAY BE DIFFERENT FROM YOUR     */ 
/*		SIMULATION MODEL. SEE THE AUTHORS  */
/*              PAPER IN BIOMETRIKA 1999.          */
/*  THE SINGLE-INDEX FUNCTION CAN BE DRAW BY THE   */
/*              OUTPUT ytheta*xtheta.              */ 
/***************************************************/ 

PRINT beta theta;
PRINT y ypred;

PRINT "THE SINGLE-INDEX FUNCTION", xtheta ytheta yr;

QUIT;
